//
// (c) 2021 wesolutions GmbH
// All rights reserved.
//

import QtQml 2.12
import QtQuick 2.12

import wesual.Ui 1.0

UiText {
    id : editableText

    property TextInput __editor
    property QtObject  validator
    property string inputMask
    property bool editOnDoubleClick : true
    property bool readOnly : false

    property Component backgroundDelegate : Rectangle {
        anchors.fill : parent || null
        anchors.margins : -2
    }

    function commit() {
        if (__editor) {
            updateRequested(__editor.text);
            __editor.destroy();
            __editor = null;
        }
    }

    function abort() {
        if (__editor) {
            __editor.destroy();
            __editor = null;
            aborted();
        }
    }

    function beginEdit(text) {
        if (readOnly)
            return;

        if (!__editor) {
            __editor = editorComponent.createObject(editableText, {
                text : editableText.text
            });
            if (text) {
                __editor.text = text;
                __editor.cursorPosition = text.length;
            }
        }
    }

    signal updateRequested(string newValue)
    signal aborted()

    color : {
        if (!editableText.enabled) {
            return UiColors.getColor(UiColors.MediumLightGrey27);
        } else if (highlight) {
            return UiColors.getColor(UiColors.White);
        } else if (editableText.readOnly) {
            return UiColors.getColor(UiColors.MediumGrey45);
        } else {
            return UiColors.getColor(UiColors.DarkGrey80);
        }
    }

    Keys.onLeftPressed : event.accepted = !!__editor
    Keys.onRightPressed : event.accepted = !!__editor

    Binding on text {
        when  : __editor
        value : __editor ? __editor.text : ""
    }

    MouseArea {
        anchors.fill : editableText
        enabled : editOnDoubleClick && !readOnly
        z : -1

        onDoubleClicked : beginEdit()
    }

    Component {
        id : editorComponent

        TextInput {
            id : editor

            property bool committed : false

            anchors.fill : editableText
            clip : true
            selectionColor : UiColors.getColor(UiColors.SelectionGreen)
            selectByMouse : true
            font : editableText.font
            color : UiColors.getColor(UiColors.DarkGrey80)
            leftPadding : editableText.leftPadding
            rightPadding : editableText.rightPadding
            topPadding : editableText.topPadding
            bottomPadding : editableText.bottomPadding
            padding : editableText.padding
            text : editableText.text
            horizontalAlignment : editableText.horizontalAlignment
            validator : editableText.validator
            inputMask : editableText.inputMask
            z : 1

            onActiveFocusChanged : {
                if (!activeFocus && !committed) {
                    editableText.commit();
                }
            }

            Component.onCompleted : {
                forceActiveFocus();
                cursorPosition = 0;
                selectAll();

                if (editableText.backgroundDelegate) {
                    editableText.backgroundDelegate.createObject(editor, {
                        parent : editableText,
                        z : 0
                    });
                }
            }

            Keys.onReturnPressed : {
                committed = true;
                editableText.commit()
            }
            Keys.onEnterPressed : {
                committed = true;
                editableText.commit()
            }

            Shortcut {
                sequence : "Esc"
                onActivated : editableText.abort()
            }

            UiTextContextMenu {
                textEditor : editor
            }
        }
    }
}
